// Copyright 2018-2021 Bret Jordan, All rights reserved.
//
// This package contains a series of changes and additions to the the build-in
// log package from the Go Authors found here:
// https://golang.org/src/log/log.go
// See their copyright below.
//
// Version 1.3.0

// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Package log implements a simple logging package. It defines a type, Logger,
// with methods for formatting output. It also has a predefined 'standard'
// Logger accessible through helper functions Print[f|ln], Fatal[f|ln], and
// Panic[f|ln], which are easier to use than creating a Logger manually.
// That logger writes to standard error and prints the date and time
// of each logged message.
// Every log message is output on a separate line: if the message being
// printed does not end in a newline, the logger will add one.
// The Fatal functions call os.Exit(1) after writing the log message.
// The Panic functions call panic after writing the log message.
package log

import (
	"fmt"
	"io"
	"os"
	"runtime"
	"strings"
	"sync"
	"time"
)

// These flags define which text to prefix to each log entry generated by the Logger.
// Bits are or'ed together to control what's printed.
// With the exception of the Lmsgprefix flag, there is no
// control over the order they appear (the order listed here)
// or the format they present (as described in the comments).
// The prefix is followed by a colon only when Llongfile or Lshortfile
// is specified.
// For example, flags Ldate | Ltime (or LstdFlags) produce,
//	2009/01/23 01:23:23 message
// while flags Ldate | Ltime | Lmicroseconds | Llongfile produce,
//	2009/01/23 01:23:23.123123 /a/b/c/d.go:23: message
const (
	Ldate         = 1 << iota     // the date in the local time zone: 2009/01/23
	Ltime                         // the time in the local time zone: 01:23:23
	Lmicroseconds                 // microsecond resolution: 01:23:23.123123.  assumes Ltime.
	Llongfile                     // full file name and line number: /a/b/c/d.go:23
	Lshortfile                    // final file name element and line number: d.go:23. overrides Llongfile
	LUTC                          // if Ldate or Ltime is set, use UTC rather than the local time zone
	Lmsgprefix                    // move the "prefix" from the beginning of the line to before the message
	LstdFlags     = Ldate | Ltime // initial values for the standard logger
)

// A Logger represents an active logging object that generates lines of
// output to an io.Writer. Each logging operation makes a single call to
// the Writer's Write method. A Logger can be used simultaneously from
// multiple goroutines; it guarantees to serialize access to the Writer.
//
// Calldepth is used to recover the PC and is provided for generality, although
// at the moment on all pre-defined paths it will be 2.
// Jordan: Added calldepth, levels, and formattedPrefix to the struct
type Logger struct {
	mu              sync.Mutex // ensures atomic writes; protects the following fields
	prefix          string     // prefix to write at beginning of each line
	flag            int        // properties
	out             io.Writer  // destination for output
	buf             []byte     // for accumulating text to write
	calldepth       int
	levels          map[string]bool
	formattedPrefix bool
}

// New creates a new Logger. The out variable sets the
// destination to which log data will be written.
// The prefix appears at the beginning of each generated log line, or
// after the log header if the Lmsgprefix flag is provided.
// The flag argument defines the logging properties.
// Jordan: Added levels and enabled fatal by default
func New(out io.Writer, prefix string, flag int) *Logger {
	l := Logger{out: out, prefix: prefix, flag: flag}
	// Since panic and fatal are part of the standard library, lets enable them
	// by default so that they just work and the checks in the Output function
	// will allow them.
	l.levels = map[string]bool{
		"panic": true,
		"fatal": true,
		"error": false,
		"warn":  false,
		"info":  false,
		"debug": false,
		"trace": false,
	}
	return &l
}

// SetOutput sets the output destination for the logger.
func (l *Logger) SetOutput(w io.Writer) {
	l.mu.Lock()
	defer l.mu.Unlock()
	l.out = w
}

var std = New(os.Stderr, "", LstdFlags)

// Default returns the standard logger used by the package-level output functions.
func Default() *Logger { return std }

// Cheap integer to fixed-width decimal ASCII. Give a negative width to avoid zero-padding.
func itoa(buf *[]byte, i int, wid int) {
	// Assemble decimal in reverse order.
	var b [20]byte
	bp := len(b) - 1
	for i >= 10 || wid > 1 {
		wid--
		q := i / 10
		b[bp] = byte('0' + i - q*10)
		bp--
		i = q
	}
	// i < 10
	b[bp] = byte('0' + i)
	*buf = append(*buf, b[bp:]...)
}

// formatHeader writes log header to buf in following order:
//   * l.prefix (if it's not blank and Lmsgprefix is unset),
//   * date and/or time (if corresponding flags are provided),
//   * file and line number (if corresponding flags are provided),
//   * l.prefix (if it's not blank and Lmsgprefix is set).
func (l *Logger) formatHeader(buf *[]byte, t time.Time, file string, line int) {
	if l.flag&Lmsgprefix == 0 {
		*buf = append(*buf, l.prefix...)
	}
	if l.flag&(Ldate|Ltime|Lmicroseconds) != 0 {
		if l.flag&LUTC != 0 {
			t = t.UTC()
		}
		if l.flag&Ldate != 0 {
			year, month, day := t.Date()
			itoa(buf, year, 4)
			*buf = append(*buf, '/')
			itoa(buf, int(month), 2)
			*buf = append(*buf, '/')
			itoa(buf, day, 2)
			*buf = append(*buf, ' ')
		}
		if l.flag&(Ltime|Lmicroseconds) != 0 {
			hour, min, sec := t.Clock()
			itoa(buf, hour, 2)
			*buf = append(*buf, ':')
			itoa(buf, min, 2)
			*buf = append(*buf, ':')
			itoa(buf, sec, 2)
			if l.flag&Lmicroseconds != 0 {
				*buf = append(*buf, '.')
				itoa(buf, t.Nanosecond()/1e3, 6)
			}
			*buf = append(*buf, ' ')
		}
	}
	if l.flag&(Lshortfile|Llongfile) != 0 {
		if l.flag&Lshortfile != 0 {
			short := file
			for i := len(file) - 1; i > 0; i-- {
				if file[i] == '/' {
					short = file[i+1:]
					break
				}
			}
			file = short
		}
		*buf = append(*buf, file...)
		*buf = append(*buf, ':')
		itoa(buf, line, -1)
		*buf = append(*buf, ": "...)
	}
	if l.flag&Lmsgprefix != 0 {
		*buf = append(*buf, l.prefix...)
	}
}

// Output writes the output for a logging event. The string s contains
// the text to print after the prefix specified by the flags of the
// Logger. A newline is appended if the last character of s is not
// already a newline. Calldepth is used to recover the PC and is
// provided for generality, although at the moment on all pre-defined
// paths it will be 2.
// Jordan: Set calldepth and allow default levels. Call depth is now in object
func (l *Logger) Output(level string, s string) error {
	// Lets set the default value to 2, just like it was before.
	if l.calldepth == 0 {
		l.calldepth = 2
	}

	// Allow the default standard log types to just work and do not output
	// log levels that are turned off. However, we need to address the "fatal"
	// loglevel and allow it, as it is part of the standard library. This was
	// done in the new function to enable fatal by default.
	if level != "" {
		if val, ok := l.levels[level]; !ok || val == false {
			return nil
		}
	}

	// Set prefix to be formatted
	var pre string
	if l.formattedPrefix {
		if level == "warn" || level == "info" {
			pre = "[" + strings.ToUpper(level) + "]  "
		} else {
			pre = "[" + strings.ToUpper(level) + "] "
		}

		l.SetPrefix(pre)
	}

	now := time.Now() // get this early.
	var file string
	var line int
	l.mu.Lock()
	defer l.mu.Unlock()
	if l.flag&(Lshortfile|Llongfile) != 0 {
		// Release lock while getting caller info - it's expensive.
		l.mu.Unlock()
		var ok bool
		// Jordan: use calldepth on object
		_, file, line, ok = runtime.Caller(l.calldepth)
		if !ok {
			file = "???"
			line = 0
		}
		l.mu.Lock()
	}
	l.buf = l.buf[:0]
	l.formatHeader(&l.buf, now, file, line)
	l.buf = append(l.buf, s...)
	if len(s) == 0 || s[len(s)-1] != '\n' {
		l.buf = append(l.buf, '\n')
	}
	_, err := l.out.Write(l.buf)
	return err
}

// Jordan: In all of these functions we are passing in the logging level now
// not the calldepth, since the calldepth is set on the object

// Printf calls l.Output to print to the logger.
// Arguments are handled in the manner of fmt.Printf.
func (l *Logger) Printf(format string, v ...interface{}) {
	l.Output("", fmt.Sprintf(format, v...))
}

// Print calls l.Output to print to the logger.
// Arguments are handled in the manner of fmt.Print.
func (l *Logger) Print(v ...interface{}) { l.Output("", fmt.Sprint(v...)) }

// Println calls l.Output to print to the logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Println(v ...interface{}) { l.Output("", fmt.Sprintln(v...)) }

// Fatal is equivalent to l.Print() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func (l *Logger) Fatal(v ...interface{}) {
	l.Output("fatal", fmt.Sprint(v...))
	os.Exit(1)
}

// Fatalf is equivalent to l.Printf() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func (l *Logger) Fatalf(format string, v ...interface{}) {
	l.Output("fatal", fmt.Sprintf(format, v...))
	os.Exit(1)
}

// Fatalln is equivalent to l.Println() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func (l *Logger) Fatalln(v ...interface{}) {
	l.Output("fatal", fmt.Sprintln(v...))
	os.Exit(1)
}

// Panic is equivalent to l.Print() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func (l *Logger) Panic(v ...interface{}) {
	s := fmt.Sprint(v...)
	l.Output("panic", s)
	panic(s)
}

// Panicf is equivalent to l.Printf() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func (l *Logger) Panicf(format string, v ...interface{}) {
	s := fmt.Sprintf(format, v...)
	l.Output("panic", s)
	panic(s)
}

// Panicln is equivalent to l.Println() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func (l *Logger) Panicln(v ...interface{}) {
	s := fmt.Sprintln(v...)
	l.Output("panic", s)
	panic(s)
}

// Flags returns the output flags for the logger.
// The flag bits are Ldate, Ltime, and so on.
func (l *Logger) Flags() int {
	l.mu.Lock()
	defer l.mu.Unlock()
	return l.flag
}

// SetFlags sets the output flags for the logger.
// The flag bits are Ldate, Ltime, and so on.
func (l *Logger) SetFlags(flag int) {
	l.mu.Lock()
	defer l.mu.Unlock()
	l.flag = flag
}

// Prefix returns the output prefix for the logger.
func (l *Logger) Prefix() string {
	l.mu.Lock()
	defer l.mu.Unlock()
	return l.prefix
}

// SetPrefix sets the output prefix for the logger.
func (l *Logger) SetPrefix(prefix string) {
	l.mu.Lock()
	defer l.mu.Unlock()
	l.prefix = prefix
}

// Writer returns the output destination for the logger.
func (l *Logger) Writer() io.Writer {
	l.mu.Lock()
	defer l.mu.Unlock()
	return l.out
}

// SetOutput sets the output destination for the standard logger.
func SetOutput(w io.Writer) {
	std.mu.Lock()
	defer std.mu.Unlock()
	std.out = w
}

// Flags returns the output flags for the standard logger.
// The flag bits are Ldate, Ltime, and so on.
func Flags() int {
	return std.Flags()
}

// SetFlags sets the output flags for the standard logger.
// The flag bits are Ldate, Ltime, and so on.
func SetFlags(flag int) {
	std.SetFlags(flag)
}

// Prefix returns the output prefix for the standard logger.
func Prefix() string {
	return std.Prefix()
}

// SetPrefix sets the output prefix for the standard logger.
func SetPrefix(prefix string) {
	std.SetPrefix(prefix)
}

// Writer returns the output destination for the standard logger.
func Writer() io.Writer {
	return std.Writer()
}

// These functions write to the standard logger.

// Print calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Print.
func Print(v ...interface{}) {
	std.Output("", fmt.Sprint(v...))
}

// Printf calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func Printf(format string, v ...interface{}) {
	std.Output("", fmt.Sprintf(format, v...))
}

// Println calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Println(v ...interface{}) {
	std.Output("", fmt.Sprintln(v...))
}

// Fatal is equivalent to Print() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func Fatal(v ...interface{}) {
	std.Output("fatal", fmt.Sprint(v...))
	os.Exit(1)
}

// Fatalf is equivalent to Printf() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func Fatalf(format string, v ...interface{}) {
	std.Output("fatal", fmt.Sprintf(format, v...))
	os.Exit(1)
}

// Fatalln is equivalent to Println() followed by a call to os.Exit(1).
// Jordan: added fatal as a level so it can show up in formatted logs
func Fatalln(v ...interface{}) {
	std.Output("fatal", fmt.Sprintln(v...))
	os.Exit(1)
}

// Panic is equivalent to Print() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func Panic(v ...interface{}) {
	s := fmt.Sprint(v...)
	std.Output("panic", s)
	panic(s)
}

// Panicf is equivalent to Printf() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func Panicf(format string, v ...interface{}) {
	s := fmt.Sprintf(format, v...)
	std.Output("panic", s)
	panic(s)
}

// Panicln is equivalent to Println() followed by a call to panic().
// Jordan: added panic as a level so it can show up in formatted logs
func Panicln(v ...interface{}) {
	s := fmt.Sprintln(v...)
	std.Output("panic", s)
	panic(s)
}

// Output writes the output for a logging event. The string s contains
// the text to print after the prefix specified by the flags of the
// Logger. A newline is appended if the last character of s is not
// already a newline. Calldepth is the count of the number of
// frames to skip when computing the file name and line number
// if Llongfile or Lshortfile is set; a value of 1 will print the details
// for the caller of Output.
// Jordan: calldepth is set on the object so passing in empty levels
func Output(level string, s string) error {
	return std.Output("level", s) // +1 for this frame.
}

// ----------------------------------------------------------------------
//
// Jordan: My changes to enable logging levels
//
// ----------------------------------------------------------------------

// SetCallDepth - This function will set the call depth. By default the call
// depth is set at 2. A depth of 2 represents the behavior of the standard
// library.
func (l *Logger) SetCallDepth(d int) {
	l.calldepth = d
}

// SetCallDepth - This function will set the call depth. By default the call
// depth is set at 2. A depth of 2 represents the behavior of the standard
// library.
func SetCallDepth(d int) {
	std.SetCallDepth(d)
}

// EnableLevel - This function will enable the output from the supplied logging
// level
func (l *Logger) EnableLevel(level string) {
	l.levels[level] = true
}

// EnableLevel - This function will enable the output from the supplied logging
// level
func EnableLevel(level string) {
	std.EnableLevel(level)
}

// DisableLevel - This function will disable the output from the supplied
// logging level
func (l *Logger) DisableLevel(level string) {
	if _, ok := l.levels[level]; ok {
		l.levels[level] = false
	}
}

// DisableLevel - This function will disable the output from the supplied
// logging level
func DisableLevel(level string) {
	std.DisableLevel(level)
}

// GetLevel - This function will return the state of a given level
func (l *Logger) GetLevel(level string) bool {
	if _, ok := l.levels[level]; ok {
		return l.levels[level]
	}
	return false
}

// GetLevel - This function will return the state of a given level
func GetLevel(level string) bool {
	return std.GetLevel(level)
}

// EnableFormattedPrefix - This function will enable the formatted prefix in output
func (l *Logger) EnableFormattedPrefix() {
	l.formattedPrefix = true
}

// EnableFormattedPrefix - This function will enable the formatted prefix in output
func EnableFormattedPrefix() {
	std.formattedPrefix = true
}

// EnableLevelsByNumber - This function will enable logging levels by number
func (l *Logger) EnableLevelsByNumber(num int) {
	switch num {
	case 1:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
	case 2:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
		l.EnableLevel("error")
	case 3:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
		l.EnableLevel("error")
		l.EnableLevel("warn")
	case 4:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
		l.EnableLevel("error")
		l.EnableLevel("warn")
		l.EnableLevel("info")
	case 5:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
		l.EnableLevel("error")
		l.EnableLevel("warn")
		l.EnableLevel("info")
		l.EnableLevel("debug")
	case 10:
		l.EnableLevel("panic")
		l.EnableLevel("fatal")
		l.EnableLevel("error")
		l.EnableLevel("warn")
		l.EnableLevel("info")
		l.EnableLevel("debug")
		l.EnableLevel("trace")
	}
	return
}

// EnableLevelsByNumber - This function will enable logging levels by number
func EnableLevelsByNumber(num int) {
	std.EnableLevelsByNumber(num)
}

// DisableAllLevels - This function will the output from all logging level
func (l *Logger) DisableAllLevels() {
	for k := range l.levels {
		l.levels[k] = false
	}
}

// DisableAllLevels - This function will the output from all logging level
func DisableAllLevels() {
	std.DisableAllLevels()
}

// ----------------------------------------------------------------------
// New logger functions and methods
// ----------------------------------------------------------------------

// Error - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func (l *Logger) Error(v ...interface{}) {
	l.Output("error", fmt.Sprint(v...))
}

// Errorf - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func (l *Logger) Errorf(format string, v ...interface{}) {
	l.Output("error", fmt.Sprintf(format, v...))
}

// Errorln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Errorln(v ...interface{}) {
	l.Output("error", fmt.Sprintln(v...))
}

// Warn - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func (l *Logger) Warn(v ...interface{}) {
	l.Output("warn", fmt.Sprint(v...))
}

// Warnf - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Printf.
func (l *Logger) Warnf(format string, v ...interface{}) {
	l.Output("warn", fmt.Sprintf(format, v...))
}

// Warnln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Warnln(v ...interface{}) {
	l.Output("warn", fmt.Sprintln(v...))
}

// Info - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func (l *Logger) Info(v ...interface{}) {
	l.Output("info", fmt.Sprint(v...))
}

// Infof - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Printf.
func (l *Logger) Infof(format string, v ...interface{}) {
	l.Output("info", fmt.Sprintf(format, v...))
}

// Infoln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Infoln(v ...interface{}) {
	l.Output("info", fmt.Sprintln(v...))
}

// Debug - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func (l *Logger) Debug(v ...interface{}) {
	l.Output("debug", fmt.Sprint(v...))
}

// Debugf - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func (l *Logger) Debugf(format string, v ...interface{}) {
	l.Output("debug", fmt.Sprintf(format, v...))
}

// Debugln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Debugln(v ...interface{}) {
	l.Output("debug", fmt.Sprintln(v...))
}

// Trace - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func (l *Logger) Trace(v ...interface{}) {
	l.Output("trace", fmt.Sprint(v...))
}

// Tracef - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func (l *Logger) Tracef(format string, v ...interface{}) {
	l.Output("trace", fmt.Sprintf(format, v...))
}

// Traceln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func (l *Logger) Traceln(v ...interface{}) {
	l.Output("trace", fmt.Sprintln(v...))
}

// Level - This function calls Output to print to the standard logger. The first
// parameter is a logging level, this allows the printing of arbitrary logging
// levels. Arguments are handled in the manner of fmt.Print.
func (l *Logger) Level(level string, v ...interface{}) {
	l.Output(level, fmt.Sprint(v...))
}

// Levelf - This function calls Output to print to the standard logger. The
// first parameter is a logging level, this allows the printing of arbitrary
// logging levels. Arguments are handled in the manner of fmt.Printf.
func (l *Logger) Levelf(level, format string, v ...interface{}) {
	l.Output(level, fmt.Sprintf(format, v...))
}

// Levelln - This function calls Output to print to the standard logger. The
// first parameter is a logging level, this allows the printing of arbitrary
// logging levels. Arguments are handled in the manner of fmt.Println.
func (l *Logger) Levelln(level string, v ...interface{}) {
	l.Output(level, fmt.Sprintln(v...))
}

// Error - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func Error(v ...interface{}) {
	std.Output("error", fmt.Sprint(v...))
}

// Errorf - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func Errorf(format string, v ...interface{}) {
	std.Output("error", fmt.Sprintf(format, v...))
}

// Errorln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Errorln(v ...interface{}) {
	std.Output("error", fmt.Sprintln(v...))
}

// Warn - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func Warn(v ...interface{}) {
	std.Output("warn", fmt.Sprint(v...))
}

// Warnf - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Printf.
func Warnf(format string, v ...interface{}) {
	std.Output("warn", fmt.Sprintf(format, v...))
}

// Warnln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Warnln(v ...interface{}) {
	std.Output("warn", fmt.Sprintln(v...))
}

// Info - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func Info(v ...interface{}) {
	std.Output("info", fmt.Sprint(v...))
}

// Infof - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Printf.
func Infof(format string, v ...interface{}) {
	std.Output("info", fmt.Sprintf(format, v...))
}

// Infoln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Infoln(v ...interface{}) {
	std.Output("info", fmt.Sprintln(v...))
}

// Debug - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func Debug(v ...interface{}) {
	std.Output("debug", fmt.Sprint(v...))
}

// Debugf - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func Debugf(format string, v ...interface{}) {
	std.Output("debug", fmt.Sprintf(format, v...))
}

// Debugln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Debugln(v ...interface{}) {
	std.Output("debug", fmt.Sprintln(v...))
}

// Trace - This function calls Output to print to the standard logger. Arguments
// are handled in the manner of fmt.Print.
func Trace(v ...interface{}) {
	std.Output("trace", fmt.Sprint(v...))
}

// Tracef - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Printf.
func Tracef(format string, v ...interface{}) {
	std.Output("trace", fmt.Sprintf(format, v...))
}

// Traceln - This function calls Output to print to the standard logger.
// Arguments are handled in the manner of fmt.Println.
func Traceln(v ...interface{}) {
	std.Output("trace", fmt.Sprintln(v...))
}

// Level - This function calls Output to print to the standard logger. The first
// parameter is a logging level, this allows the printing of arbitrary logging
// levels. Arguments are handled in the manner of fmt.Print.
func Level(level string, v ...interface{}) {
	std.Output(level, fmt.Sprint(v...))
}

// Levelf - This function calls Output to print to the standard logger. The
// first parameter is a logging level, this allows the printing of arbitrary
// logging levels. Arguments are handled in the manner of fmt.Printf.
func Levelf(level, format string, v ...interface{}) {
	std.Output(level, fmt.Sprintf(format, v...))
}

// Levelln - This function calls Output to print to the standard logger. The
// first parameter is a logging level, this allows the printing of arbitrary
// logging levels. Arguments are handled in the manner of fmt.Println.
func Levelln(level string, v ...interface{}) {
	std.Output(level, fmt.Sprintln(v...))
}

// ----------------------------------------------------------------------
