# -*- coding: utf-8 -*-
#
# Licensed under the terms of the BSD 3-Clause
# (see plotpy/LICENSE for details)

"""
Testing image coordinates issues

Check that the first image pixel is centered on (0, 0) coordinates.

See https://github.com/PlotPyStack/guiqwt/issues/90
"""

# guitest: show

from __future__ import annotations

from typing import Literal

import numpy as np
from guidata.qthelpers import qt_app_context

from plotpy.builder import make
from plotpy.tests import data as ptd
from plotpy.tests import vistools as ptv
from plotpy.tools import DisplayCoordsTool


def test_pixel_coords(image_type: Literal["standard", "xy"] = "standard") -> None:
    """Testing image pixel coordinates"""
    title = test_pixel_coords.__doc__ + f" ({image_type} image)"
    data = ptd.gen_2d_gaussian(20, np.uint8, x0=-10, y0=-10, mu=7, sigma=10.0)
    with qt_app_context(exec_loop=True):
        if image_type == "xy":
            x = np.linspace(0.0, 10.0, data.shape[1], dtype=float)
            y = np.linspace(0.0, 10.0, data.shape[0], dtype=float) ** 2 / 10.0
            image = make.xyimage(x, y, data, interpolation="nearest")
        else:
            image = make.image(data, interpolation="nearest")
        text = "First pixel should be centered on (0, 0) coordinates"
        label = make.label(text, (1.0, 1.0), (0, 0), "L")
        rect = make.rectangle(5.0, 5.0, 10.0, 10.0, "Rectangle")
        cursors = []
        for i_cursor in range(0, 21, 10):
            cursors.append(make.vcursor(i_cursor, movable=False))
            cursors.append(make.hcursor(i_cursor, movable=False))
        win = ptv.show_items([image, label, rect] + cursors, wintitle=title)
        plot = win.get_plot()
        plot.select_item(image)
        win.manager.get_tool(DisplayCoordsTool).activate_curve_pointer(True)


if __name__ == "__main__":
    test_pixel_coords("standard")
    test_pixel_coords("xy")
