/* 
 * Copyright (C) 2000-2003 by Oswald Buddenhagen <puf@ossi.cjb.net>
 * based on puf 0.1.x (C) 1999,2000 by Anders Gavare <gavare@hotmail.com>
 *
 * You may modify and distribute this code under the terms of the GPL.
 * There is NO WARRANTY of any kind. See COPYING for details.
 *
 * hostlist.c - manage the hostname cache
 *
 */

#include "puf.h"

int always_primary_name;

static host_t *hostlist;	/*  list of known hosts  */
static int real_num_hosts;

whost_t *queue_dns_lookup;	/*  waiting for start of lookup  */
dnsproc_t *list_dns_busy, *list_dns_idle;	/*  helper processes  */


#define hhash(hp) calc_hash(hp->name)

/*  create new hostname structure  */
static host_t *
add_host(const char *name, int len)
{
    host_t *h;
    int hash;

    if ((h = mmalloc(sizeof(host_t) + len))) {
	memcpy(h->name, name, len);
	hash = calc_nhash(name, len - 1);
	h_add(hostlist, real_num_hosts, host_t, h, hash, free(h); return 0;, hhash);
    }
    return h;
}


/*  tolower() on whole string  */
static int 
lcase(char *d, const char *s)
{
    int l;

    for (l = 0; (d[l] = tolower((int)s[l])); l++);
    return l + 1;
}


/*  FIXME: maybe, we should use inet_aton (or inet_addr) before doing 
    the gethostbyname magic. this would save the forks for numerical
    input. also, gethostbyname reportedly fails addresses like
    http://2165339403/~f98anga/ on openbsd  */

/*  return a cached host entry for the given host  */
host_t *
host_lookup_fast(const char *name, int namlen)
{
    int hash = calc_nhash(name, namlen);

#define hcmp(h) !memcmp(name, h->name, namlen + 1)
#define hfnd(h) return h;

    h_find(hostlist, real_num_hosts, host_t, hash, hcmp, hfnd);
    return 0;
}


/*  starts an asynchronous dns lookup for the given host  */
/*  the first name referred to determines the local directory 
    name for this host.  */
whost_t *
host_lookup_full(const char *name, int namlen)
{
    host_t *h;
    whost_t *nwh;

    dbg(DNS, ("host_lookup_full for '%.*s'\n", namlen, name));

    if (!(nwh = mmalloc(sizeof(*nwh))))
	return 0;

    if (!(h = add_host(name, namlen + 1))) {
	free(nwh);
	return 0;
    }

    h->info = (hinfo_t *)nwh;
    h->ready = 0;

    nwh->urls = 0;
    nwh->num_proxies = 0;
    nwh->host = h;
    cq_append(queue_dns_lookup, nwh); /* XXX optimize idle case */

    return nwh;
}

int 
start_lookup(dnsproc_t *proc)
{
    int l = strlen(proc->whost->host->name) + 1;
    dbg(DNS, ("starting dns lookup, helper %d\n", proc->pid));
    if (send(proc->fd, proc->whost->host->name, l, 0) != l) {
	prx(ERR, "cannot talk to DNS helper!\n");
	return 0;
    }
    return 1;
}

int 
finish_lookup(dnsproc_t *proc)
{
    whost_t *wh;
    host_t *h;
    hinfo_t *hi;
    int i, na, cp, nhs, hapi;
    host_t *hs[16];
    u_char buf[1024];

    dbg(DNS, ("finishing dns lookup, helper %d\n", proc->pid));
    if (recv(proc->fd, buf, sizeof(buf), 0) < (int)sizeof(int)) {
	prx(ERR, "cannot talk to DNS helper!\n");
	return 0;
    }
    wh = proc->whost;
    if (!wh->host)
	return 1;
    dbg(DNS, (" request originator was %s\n", wh->host->name));
    na = ((int *)buf)[0];
    if (!na) {
	prx(ERR, "DNS lookup for '%s' failed!\n", wh->host->name);
	goto badhost;
    }
    if (na < 0) {
	prx(ERR, "DNS lookup for '%s' timed out!\n", wh->host->name);
	goto badhost;
    }
    if (((int *)buf)[1] != sizeof(struct in_addr)) {
	prx(ERR, "cannot handle address returned for '%s'!\n", wh->host->name);
	goto badhost;
    }

    for (cp = 2 * sizeof(int) + na * sizeof(struct in_addr),
	 nhs = 0, hi = 0, hapi = 0;
	 buf[cp]; cp += buf[cp] + 1)
    {
	if (nhs >= (int)(sizeof(hs)/sizeof(hs[0])) - 1) {
	    prx(WRN, "lookup of '%s' yielded too many aliases\n", wh->host->name);
	    break;
	}

	if (!memcmp(wh->host->name, buf + cp + 1, buf[cp])) {
	    dbg(DNS, (" entry %d is origin\n", nhs));
	    hapi = 1;
	    hs[nhs++] = wh->host;
	} else if ((h = host_lookup_fast(buf + cp + 1, buf[cp] - 1))) {
	    if (h->ready) {
		dbg(DNS, (" entry %d (%s) is already resolved\n",
			  nhs, h->name));
		if (hi && h->info != hi) {
		    prx(WRN, "inconsistent DNS info for '%s'\n", h->name);
		    continue;
		}
		hi = h->info;
	    } else
		dbg(DNS, (" entry %d (%s) has resolution pending\n",
			  nhs, h->name));
	    hs[nhs++] = h;
	} else {
	    dbg(DNS, (" entry %d (%s) is new\n", nhs, buf + cp + 1));
	    if (!(h = add_host(buf + cp + 1, buf[cp]))) {
		if (!nhs)
		    goto badhost;
		continue;
	    } else {
		h->ready = 0;
		h->info = 0;
		hs[nhs++] = h;
	    }
	}
    }

    if (!hapi)
	hs[nhs++] = wh->host;

    /*  possibly create new hostinfo structure ...  */
  if (!hi && (hi = mmalloc(sizeof(hinfo_t) + na * sizeof(haddr_t)))) {
    /*  ... and initialize it  */
    hi->name = hs[0]->name;
    hi->lname = always_primary_name ? hs[0]->name : wh->host->name;
    hi->is_http11 = 1;
    hi->cur_ip = 0;
    hi->num_ips = na;
    /*  copy list of ip-addresses and initialize retry counters  */
    for (i = 0; i < na; i++) {
	hi->ips[i].addr = ((struct in_addr *)(((int *)buf) + 2))[i];
	hi->ips[i].retry_time = 0;
	hi->ips[i].last_errt = 0;
	hi->ips[i].attempt = 0;
    }
  }

    for (i = 0; i < nhs; i++)
	if (!hs[i]->ready) {
	    wh = (whost_t *)hs[i]->info;
	    hs[i]->info = hi;
	    hs[i]->ready = 1;
	    if (wh)
		finish_whost(wh);
	}

    return 1;

badhost:
    wh->host->info = 0;
    wh->host->ready = 1;
    finish_whost(wh);
    return 1;
}

#include <setjmp.h>

jmp_buf alrmjmp;

static void
sigalrm(int n)
{
    (void)n;
    longjmp(alrmjmp, 1);
}

dnsproc_t *
fork_dnsproc()
{
    dnsproc_t *proc;
    struct hostent *he;
    int hl, i, na, pid, fds[2];
    u_int cp;
    sigset_t ss, oss; 
    char buf[1024];

    dbg(DNS, ("forking new dns helper\n"));

    if (socketpair(PF_UNIX, SOCK_STREAM, 0, fds) &&
	(!free_fd(0) ||
	 (socketpair(PF_UNIX, SOCK_STREAM, 0, fds) &&
	  (!free_fd(0) ||
           socketpair(PF_UNIX, SOCK_STREAM, 0, fds)))))
	return 0;

    sigfillset(&ss);
    sigprocmask(SIG_SETMASK, &ss, &oss);
    if ((pid = fork())) {
	sigprocmask(SIG_SETMASK, &oss, 0);
	close(fds[1]);
	if (pid < 0) {
            dbg(DNS, ("  failed\n"));
	    close(fds[0]);
	    return 0;
	} else {
            dbg(DNS, ("  pid = %d\n", pid));
	    if (!(proc = mmalloc(sizeof(*proc)))) {
		kill(pid, SIGTERM);
		waitpid(pid, 0, 0);
		close(fds[0]);
		return 0;
	    }
	    proc->fd = fds[0];
	    proc->pid = pid;
	    return proc;
	}
    }
    signal(SIGTERM, SIG_DFL);
    signal(SIGINT, SIG_DFL);
    sigprocmask(SIG_SETMASK, &oss, 0);
    close(fds[0]);

    for (;;) {
	if (!setjmp(alrmjmp)) {
            dbg(DNS, ("dns helper %d: awaiting request\n", getpid()));
	    if (read(fds[1], buf, sizeof(buf)) <= 0) {
	      if (getppid() != 1)
		prx(ERR, "DNS helper: cannot read control socket!\n");
		exit(1);
	    }
            dbg(DNS, ("dns helper %d: looking up '%s'\n", getpid(), buf));
	    signal(SIGALRM, (void (*)(int))sigalrm);
	    alarm(timeout_dns);
	    he = gethostbyname(buf);
	    alarm(0);
	    if (!he) {
                dbg(DNS, ("dns helper %d: lookup failed\n", getpid()));
		((int *)buf)[0] = 0;
		if (write(fds[1], buf, sizeof(int)) != sizeof(int)) {
		  if (getppid() != 1)
		    prx(ERR, "DNS helper: cannot write control socket!\n");
		    exit(1);
		}
	    } else {
		/*  count ip-addresses for this name  */
		for (na = 0, cp = 2 * sizeof(int); he->h_addr_list[na]; 
		     na++, cp += he->h_length)
		    memcpy(buf + cp, he->h_addr_list[na], he->h_length);
                dbg(DNS, ("dns helper %d: lookup successful, %d addresses\n", getpid(), na));
		((int *)buf)[0] = na;
		((int *)buf)[1] = he->h_length;

		/*  copy name and aliases  */
		hl = lcase(buf + cp + 1, he->h_name);
		buf[cp] = hl;
		cp += hl + 1;
		for (i = 0; he->h_aliases[i]; i++) {
		    hl = lcase(buf + cp + 1, he->h_aliases[i]);
		    buf[cp] = hl;
		    cp += hl + 1;
		}
		buf[cp++] = 0;

		if (write(fds[1], buf, cp) != (int)cp) {
		  if (getppid() != 1)
		    prx(ERR, "DNS helper: cannot write control socket!\n");
		    exit(1);
		}
	    }
	} else {
            dbg(DNS, ("dns helper %d: lookup timed out\n", getpid()));
	    ((int *)buf)[0] = -1;
	    if (write(fds[1], buf, sizeof(int)) != sizeof(int)) {
	      if (getppid() != 1)
		prx(ERR, "DNS helper: cannot write control socket!\n");
		exit(1);
	    }
	}
    }
}

void
reap_dnsproc(dnsproc_t *proc)
{
    dbg(DNS, ("reaping dns helper %d\n", proc->pid));
    kill(proc->pid, SIGTERM);
    waitpid(proc->pid, 0, 0);
    close(proc->fd);
    free(proc);
}
