\name{utilities}

\alias{utilities}
\alias{moveDLL}
\alias{moveDLL-methods}
\alias{moveDLL,CFunc-method}
\alias{writeCFunc}
\alias{readCFunc}
\alias{print,CFunc-method}
\alias{print,CFuncList-method}
\alias{code}
\alias{code-methods}
\alias{code,character-method}
\alias{code,CFunc-method}
\alias{code,CFuncList-method}

\title{ Printing, reading and writing compiled function objects }

\description{
  \code{moveDLL} moves the DLL used by a compiled function to a user defined
  location.

  \code{writeCFunc} saves a \code{CFunc} object after the DLL has been moved to
  the desired location using \code{moveDLL}.

  \code{readCFunc} reads a \code{CFunc} object that has been saved using
  \code{writeCFunc}.

  The \code{print} and \code{code} methods respectively print the entire
  object or only the code parts.
}

\usage{
moveDLL(x, ...)
\S4method{moveDLL}{CFunc}(x, name, directory, unload = FALSE, overwrite = FALSE, verbose = FALSE)

writeCFunc(x, file)
readCFunc(file)

\S4method{print}{CFunc}(x)
\S4method{print}{CFuncList}(x)

\S4method{code}{CFunc}(x, linenumbers = TRUE)
\S4method{code}{CFuncList}(x, linenumbers = TRUE)
}

\arguments{

  \item{x}{A \code{CFunc} or \code{CFuncList} object as created by \code{\link{cfunction}}}

  \item{name}{The base of the file name that the DLL should be moved to. The file name
    extension will depend on the operating system used}

  \item{directory}{The directory that the DLL should be written to}

  \item{unload}{In case the new path constructed from \code{name} and
    \code{directory} points to a loaded DLL, should we unload it?}

  \item{overwrite}{In case there is a file at the new path constructed from
    \code{name} and \code{directory} should we overwrite that file?}

  \item{verbose}{Should we print a message stating where the DLL was copied
    if the operation was successful?}

  \item{file}{The file path for writing and reading the object generated by
    \code{\link{cfunction}}. Consider using a file name extension like
    \code{.rda} or \code{.RData} to indicate that this is a serialized
    R object.
  }

  \item{linenumbers}{If \code{TRUE} all code lines will be numbered.}

  \item{\dots}{May be used in future methods}

}

\value{

  Function \code{readDynLib} returns a \code{CFunc} object.

  Function \code{writeDynLib} returns the name of the \code{.CFunc} file that
  was created.
}

\details{

  If you move the DLL to a user defined location with \code{moveDLL}, this will
  prevent removal of the DLL at garbage collection and, if not written to the
  session \code{\link{tempdir}}, removal at session termination. However,
  saving and reloading an object will still loose the pointer to the DLL.

  Only if their DLL has been moved, \code{CFunc} objects can be saved by
  \code{writeCFunc} and restored by \code{readCFunc}.

}

\note{

  \itemize{

    \item The code of a \code{CFunc} or \code{CFuncList} object \code{x} can be extracted
      (rather than printed), using:

    \code{x@code}.

    \item To write the code to a file (here called \code{"fn"}), without the
      new-line character \code{"\n"}:

    \code{write (strsplit(x, "\n")[[1]], file = "fn")}

  }

}

\seealso{

  \code{\link{getDynLib}}

}

\examples{

x <- as.numeric(1:10)
n <- as.integer(10)

code <- "
      integer i
      do 1 i=1, n(1)
    1 x(i) = x(i)**3
"
cubefn <- cfunction(signature(n="integer", x="numeric"), code,
  convention=".Fortran")
code(cubefn)

cubefn(n, x)$x

moveDLL(cubefn, name = "cubefn", directory = tempdir())
path <- file.path(tempdir(), "cubefn.rda")
writeCFunc(cubefn, path)
rm(cubefn)

cfn <- readCFunc(path)
cfn(3, 1:3)$x

}

\author{
  Karline Soetaert and Johannes Ranke
}

\keyword{file}
