/*
 * Unix SMB/CIFS implementation.
 *
 * Copyright (C) 2018-2019 Andreas Schneider <asn@samba.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <setjmp.h>
#include <cmocka.h>

#include "includes.h"
#include "libcli/auth/libcli_auth.h"
#include "rpc_client/init_samr.h"

#define PASSWORD "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"

static const uint8_t encrypted_test_blob[] = {
	0x37, 0x8e, 0x1d, 0xd5, 0xd3, 0x9f, 0xca, 0x8e,
	0x2f, 0x2d, 0xee, 0xc3, 0xb5, 0x50, 0xcd, 0x4e,
	0xc9, 0x08, 0x04, 0x68, 0x32, 0xc3, 0xac, 0x8e,
	0x53, 0x69, 0xd6, 0xb7, 0x56, 0xcc, 0xc0, 0xbe,
	0x4e, 0x96, 0xa7, 0x74, 0xe9, 0xaa, 0x10, 0x3d,
	0xd5, 0x8c, 0xaa, 0x12, 0x56, 0xb6, 0xf1, 0x85,
	0x21, 0xfa, 0xe9, 0xa1, 0x76, 0xe6, 0xa5, 0x33,
	0x33, 0x2f, 0x47, 0x29, 0xd6, 0xbd, 0xde, 0x64,
	0x4d, 0x15, 0x3e, 0x6a, 0x11, 0x9b, 0x52, 0xbf,
	0x7e, 0x3a, 0xeb, 0x1c, 0x55, 0xd1, 0xb2, 0xa4,
	0x35, 0x03, 0x6c, 0x39, 0x61, 0x28, 0x98, 0xc3,
	0x2d, 0xd4, 0x70, 0x69, 0x8b, 0x83, 0xe9, 0x62,
	0xbe, 0xd8, 0x72, 0x4e, 0xdf, 0xd4, 0xe9, 0xe3,
	0x46, 0x2a, 0xf9, 0x3c, 0x0f, 0x41, 0x62, 0xe1,
	0x43, 0xf0, 0x91, 0xbe, 0x72, 0xa0, 0xc9, 0x08,
	0x73, 0x20, 0x1f, 0x0d, 0x68, 0x2e, 0x32, 0xa1,
	0xb8, 0x9b, 0x08, 0xa1, 0xb4, 0x81, 0x6b, 0xf1,
	0xde, 0x0c, 0x28, 0x34, 0xe2, 0x65, 0x62, 0x54,
	0xeb, 0xc0, 0x71, 0x14, 0xad, 0x36, 0x43, 0x0e,
	0x92, 0x4d, 0x11, 0xe8, 0xdd, 0x2d, 0x5f, 0x05,
	0xff, 0x07, 0xda, 0x81, 0x4e, 0x27, 0x42, 0xa8,
	0xa9, 0x64, 0x4c, 0x74, 0xc8, 0x05, 0xbb, 0x83,
	0x5a, 0xd9, 0x90, 0x3e, 0x0d, 0x9d, 0xe5, 0x2f,
	0x08, 0xf9, 0x1b, 0xbd, 0x26, 0xc3, 0x0d, 0xac,
	0x43, 0xd5, 0x17, 0xf2, 0x61, 0xf5, 0x74, 0x9b,
	0xf3, 0x5b, 0x5f, 0xe1, 0x8a, 0xa6, 0xfd, 0xdf,
	0xff, 0xb5, 0x8b, 0xf1, 0x26, 0xf7, 0xe0, 0xa7,
	0x4f, 0x5b, 0xb8, 0x6d, 0xeb, 0xf6, 0x52, 0x68,
	0x8d, 0xa3, 0xd4, 0x7f, 0x56, 0x43, 0xaa, 0xec,
	0x58, 0x47, 0x03, 0xee, 0x9b, 0x59, 0xd9, 0x78,
	0x9a, 0xfb, 0x9e, 0xe9, 0xa6, 0x61, 0x4e, 0x6d,
	0x92, 0x35, 0xd3, 0x37, 0x6e, 0xf2, 0x34, 0x39,
	0xd4, 0xd2, 0xeb, 0xcf, 0x1c, 0x10, 0xb3, 0x2b,
	0x3e, 0x07, 0x42, 0x3e, 0x20, 0x90, 0x07, 0x3e,
	0xc7, 0xed, 0xd4, 0xdf, 0x50, 0xe5, 0xff, 0xaf,
	0x05, 0xce, 0x29, 0xbe, 0x01, 0xf8, 0xb0, 0x30,
	0x96, 0xae, 0x1b, 0x62, 0x23, 0x93, 0x91, 0x1a,
	0x52, 0x98, 0xd9, 0x59, 0xb8, 0x11, 0xec, 0xb8,
	0xcf, 0x20, 0x32, 0x90, 0x9e, 0xf2, 0x06, 0x43,
	0xb8, 0x36, 0xe3, 0x33, 0x4e, 0x6f, 0x75, 0xeb,
	0xf7, 0x6c, 0xac, 0x06, 0x5f, 0x24, 0x8e, 0x4a,
	0x03, 0xdf, 0x50, 0x31, 0xaa, 0x91, 0xd5, 0x85,
	0x95, 0x78, 0x5b, 0xf4, 0x7f, 0x3e, 0xbc, 0x41,
	0xfa, 0x10, 0xd3, 0x0f, 0x86, 0x8b, 0x23, 0xed,
	0xfc, 0xcc, 0x3e, 0x7d, 0x8c, 0xb4, 0x7c, 0xec,
	0x04, 0x7d, 0x12, 0x53, 0xa1, 0x30, 0xc5, 0xac,
	0xf3, 0x1e, 0x54, 0x1f, 0x97, 0x05, 0x86, 0x74,
	0x51, 0x13, 0x45, 0x98, 0xb8, 0x50, 0x62, 0x18,
	0x0f, 0x6d, 0x66, 0xa4, 0x88, 0x31, 0x76, 0xa3,
	0xb0, 0x75, 0x55, 0x44, 0x18, 0x7c, 0x67, 0xc7,
	0x09, 0x9c, 0xab, 0x53, 0x49, 0xc0, 0xc9, 0x27,
	0x53, 0xa6, 0x99, 0x01, 0x10, 0x49, 0x67, 0x8e,
	0x5b, 0x12, 0x96, 0x40, 0x16, 0x39, 0x11, 0x53,
	0x44, 0x8f, 0xa9, 0xbe, 0x84, 0xbe, 0xe0, 0x45,
	0xe3, 0xfd, 0x48, 0x46, 0x43, 0x53, 0x13, 0x5f,
	0xfa, 0xcf, 0x09, 0x67, 0x90, 0xa3, 0x94, 0xaa,
	0x0d, 0x1f, 0xc2, 0xc3, 0xd4, 0x7e, 0xc8, 0x14,
	0xbe, 0x84, 0xa5, 0x55, 0xee, 0x49, 0x8e, 0x03,
	0x1d, 0xaf, 0xad, 0x65, 0x2f, 0xf0, 0xd5, 0x90,
	0x5e, 0x8d, 0x29, 0x40, 0xba, 0x57, 0x26, 0xa8,
	0x6c, 0x4b, 0x59, 0x40, 0x4e, 0xc2, 0xc4, 0x88,
	0x0a, 0x06, 0x2b, 0x6c, 0x2a, 0xc7, 0x3f, 0xfe,
	0x37, 0x2c, 0x41, 0x58, 0xfe, 0x7e, 0xaf, 0xd1,
	0xd9, 0xd2, 0x9c, 0xd7, 0x8a, 0x01, 0x0e, 0x8c,
	0x9e, 0x8b, 0x5d, 0x72, 0x54, 0x00, 0xbe, 0xb2,
	0x9a, 0xc7, 0xfd, 0x83, 0x1e, 0x9c, 0x79, 0xdd,
	0x15, 0x13, 0xdc, 0x15,
};


static const uint8_t encrypted_wkssvc_test_blob[] = {
	0x13, 0x79, 0x1f, 0x1a, 0x02, 0x15, 0x72, 0x1c,
	0xa6, 0x26, 0x37, 0xeb, 0x1d, 0x41, 0x7f, 0x76,
	0x11, 0x3f, 0x49, 0x4c, 0xf9, 0x69, 0x17, 0xc8,
	0x90, 0x92, 0x53, 0xb9, 0x3f, 0xcd, 0x06, 0xfe,
	0x5c, 0x17, 0x82, 0xbd, 0x86, 0xab, 0x49, 0xee,
	0x61, 0x76, 0x55, 0xc0, 0x10, 0x51, 0xcd, 0xd9,
	0x6f, 0x12, 0x86, 0xc6, 0x19, 0x59, 0x9a, 0x2f,
	0x27, 0x1d, 0x99, 0x30, 0x60, 0x0d, 0x65, 0xc6,
	0x43, 0xd6, 0xda, 0x6b, 0x66, 0x95, 0xd4, 0xca,
	0xf5, 0x04, 0xf7, 0x01, 0x5a, 0x55, 0xb0, 0x5e,
	0x72, 0x8a, 0x75, 0xe5, 0x33, 0x4c, 0xd8, 0xc4,
	0x0e, 0xf4, 0x6d, 0x23, 0xdd, 0x05, 0x90, 0xff,
	0xe0, 0x91, 0x7b, 0x62, 0x86, 0xee, 0x78, 0x31,
	0x07, 0xad, 0x8b, 0xf9, 0xdf, 0x6f, 0x8b, 0xbd,
	0x15, 0xde, 0x1b, 0xae, 0x84, 0x68, 0xe5, 0x41,
	0x7a, 0xe3, 0x47, 0x99, 0xba, 0x61, 0xe5, 0x51,
	0x64, 0x9a, 0xa0, 0x41, 0x44, 0xa1, 0x3a, 0x52,
	0x59, 0x7d, 0x6c, 0xcf, 0xcc, 0xf0, 0x11, 0xbc,
	0xb7, 0x51, 0xa9, 0xd8, 0xfd, 0xbf, 0x58, 0x77,
	0x28, 0x86, 0xa1, 0x27, 0x94, 0xe5, 0xf6, 0x1a,
	0x6b, 0x76, 0xf7, 0x72, 0x6e, 0x17, 0x09, 0xd8,
	0x3c, 0x6f, 0x39, 0x91, 0xea, 0x48, 0x98, 0xdc,
	0x1d, 0x50, 0x2e, 0x02, 0x6e, 0x7f, 0x80, 0x5d,
	0x6e, 0x96, 0xe1, 0xcf, 0x8b, 0x6b, 0xb6, 0xed,
	0xb4, 0x6a, 0x08, 0xd2, 0x45, 0x09, 0x88, 0x86,
	0x32, 0x58, 0xd8, 0x5e, 0x33, 0x8c, 0x29, 0x1a,
	0x8f, 0xc5, 0x54, 0x9b, 0xa8, 0x32, 0xb2, 0xc1,
	0x72, 0x14, 0x6c, 0x5d, 0x9d, 0xd3, 0xf2, 0x6c,
	0x6e, 0xa4, 0x84, 0x52, 0x26, 0x73, 0x7a, 0x30,
	0x56, 0x75, 0xef, 0xd1, 0x9d, 0xcd, 0xb7, 0x87,
	0xa9, 0x5c, 0xaf, 0xe6, 0xda, 0x1d, 0x3c, 0x9c,
	0xa3, 0xb1, 0x03, 0xb0, 0x8e, 0xf6, 0xc8, 0x8f,
	0x57, 0x1c, 0xce, 0x05, 0x54, 0x99, 0xf1, 0xf9,
	0x35, 0xe6, 0xf7, 0x67, 0x94, 0xb2, 0x67, 0x5b,
	0xe7, 0xa0, 0xa2, 0x1e, 0xa2, 0x74, 0xd3, 0x99,
	0x9c, 0xd5, 0xd9, 0x90, 0x86, 0x24, 0x0e, 0x1a,
	0x0d, 0xc8, 0x9e, 0x68, 0x4c, 0x43, 0x2f, 0x42,
	0xb1, 0x7c, 0xce, 0x1e, 0xb6, 0xac, 0x56, 0xb0,
	0x8d, 0x93, 0xf1, 0x53, 0x7d, 0x0e, 0x00, 0x46,
	0xba, 0x2e, 0x14, 0x7a, 0x0b, 0xaa, 0xcb, 0x07,
	0x9b, 0x09, 0x05, 0xa0, 0xd3, 0xa1, 0x80, 0xc2,
	0xd3, 0x59, 0x92, 0x27, 0x66, 0x1f, 0xdd, 0x76,
	0x36, 0xb3, 0x3c, 0xeb, 0xd7, 0x61, 0x94, 0xb1,
	0xf8, 0x3a, 0xe0, 0xba, 0x91, 0x0f, 0xef, 0x72,
	0x2b, 0x26, 0xc6, 0xb8, 0x6d, 0x0b, 0xdb, 0x60,
	0xf8, 0xb4, 0x98, 0xd7, 0x8b, 0x8d, 0xfb, 0xa7,
	0x4e, 0x27, 0xeb, 0x00, 0xe8, 0xf7, 0x5a, 0xec,
	0xf5, 0x60, 0x28, 0x37, 0xb2, 0xc4, 0x13, 0x48,
	0x2a, 0xe1, 0x34, 0xb2, 0x53, 0xcb, 0x44, 0x34,
	0x08, 0x7e, 0x56, 0x5c, 0x2b, 0x9b, 0xe2, 0xca,
	0x90, 0xb0, 0x57, 0xee, 0x10, 0x88, 0x39, 0x84,
	0xc6, 0x66, 0x07, 0x50, 0x63, 0xcc, 0x2a, 0x7c,
	0x99, 0x8c, 0x05, 0xf9, 0xf0, 0xb8, 0x62, 0xf0,
	0x92, 0xf7, 0x2a, 0x4a, 0x17, 0x14, 0x78, 0xa1,
	0x71, 0xb6, 0x42, 0xf0, 0x87, 0xa8, 0xa4, 0x48,
	0xeb, 0xdb, 0xed, 0x8a, 0x15, 0x19, 0x1a, 0xd9,
	0xfe, 0x6f, 0x07, 0x93, 0x5d, 0x39, 0xe8, 0x0e,
	0x47, 0xe6, 0x7a, 0x7d, 0x52, 0x2e, 0x40, 0x6f,
	0x31, 0x1b, 0xf6, 0x0c, 0xc2, 0x83, 0xae, 0xc1,
	0xf0, 0xf5, 0x71, 0xcd, 0xe2, 0xf5, 0x19, 0xb6,
	0xd8, 0xb0, 0x4d, 0xa9, 0x51, 0x1c, 0xb4, 0xaf,
	0x69, 0x9a, 0x89, 0xb6, 0x5b, 0x4d, 0xfa, 0x1b,
	0xca, 0xc8, 0x61, 0x92, 0x3a, 0xd6, 0x76, 0xad,
	0x5d, 0xa6, 0x17, 0x60, 0x3e, 0xea, 0x94, 0xcf,
	0x6d, 0x1b, 0x98, 0x5c, 0x19, 0x9e, 0x4e, 0xd3,
	0x21, 0x55, 0xda, 0xe3,
};

static void torture_decode_rc4_passwd_buffer(void **state)
{
	char *password_decoded = NULL;
	size_t password_decoded_len = 0;
	DATA_BLOB session_key = data_blob_const("SystemLibraryDTC", 16);
	struct samr_CryptPasswordEx out_pwd_buf = {
		.data = {0},
	};
	NTSTATUS status;
	bool ok;

	memcpy(out_pwd_buf.data,
	       encrypted_test_blob,
	       sizeof(out_pwd_buf.data));

	status = decode_rc4_passwd_buffer(&session_key, &out_pwd_buf);
	assert_true(NT_STATUS_IS_OK(status));

	ok = decode_pw_buffer(NULL,
			      out_pwd_buf.data,
			      &password_decoded,
			      &password_decoded_len,
			      CH_UTF16);
	assert_true(ok);
	assert_int_equal(password_decoded_len, strlen(PASSWORD));
	assert_string_equal(password_decoded, PASSWORD);
}

static void torture_rc4_passwd_buffer(void **state)
{
	char *password_decoded = NULL;
	size_t password_decoded_len = 0;
	DATA_BLOB session_key = data_blob_const("SystemLibraryDTC", 16);
	struct samr_CryptPasswordEx out_pwd_buf = {
		.data = {0},
	};
	NTSTATUS status;
	bool ok;

	status = init_samr_CryptPasswordEx(PASSWORD,
					   &session_key,
					   &out_pwd_buf);
	assert_true(NT_STATUS_IS_OK(status));

	status = decode_rc4_passwd_buffer(&session_key, &out_pwd_buf);
	assert_true(NT_STATUS_IS_OK(status));

	ok = decode_pw_buffer(NULL,
			      out_pwd_buf.data,
			      &password_decoded,
			      &password_decoded_len,
			      CH_UTF16);
	assert_true(ok);
	assert_int_equal(password_decoded_len, strlen(PASSWORD));
	assert_string_equal(password_decoded, PASSWORD);
	talloc_free(password_decoded);
}

static void torture_endode_decode_rc4_passwd_buffer(void **state)
{
	char *password_decoded = NULL;
	size_t password_decoded_len = 0;
	DATA_BLOB session_key = data_blob_const("SystemLibraryDTC", 16);
	struct samr_CryptPasswordEx out_pwd_buf = {
		.data = {0},
	};
	NTSTATUS status;
	bool ok;

	status = encode_rc4_passwd_buffer(PASSWORD,
					  &session_key,
					  &out_pwd_buf);
	assert_true(NT_STATUS_IS_OK(status));

	status = decode_rc4_passwd_buffer(&session_key, &out_pwd_buf);
	assert_true(NT_STATUS_IS_OK(status));

	ok = decode_pw_buffer(NULL,
			      out_pwd_buf.data,
			      &password_decoded,
			      &password_decoded_len,
			      CH_UTF16);
	assert_true(ok);
	assert_int_equal(password_decoded_len, strlen(PASSWORD));
	assert_string_equal(password_decoded, PASSWORD);
	talloc_free(password_decoded);
}

static void torture_decode_wkssvc_join_password_buffer(void **state)
{
	DATA_BLOB session_key = data_blob_const("SystemLibraryDTC", 16);
	struct wkssvc_PasswordBuffer pwd_buf = {
		.data = {0},
	};
	char *password_decoded = NULL;
	TALLOC_CTX *mem_ctx = NULL;
	WERROR werr;

	mem_ctx = talloc_new(NULL);
	assert_non_null(mem_ctx);

	memcpy(pwd_buf.data,
	       encrypted_wkssvc_test_blob,
	       sizeof(pwd_buf.data));

	werr = decode_wkssvc_join_password_buffer(mem_ctx,
						  &pwd_buf,
						  &session_key,
						  &password_decoded);
	assert_true(W_ERROR_IS_OK(werr));
	assert_non_null(password_decoded);
	assert_string_equal(password_decoded, PASSWORD);

	TALLOC_FREE(mem_ctx);
}

static void torture_wkssvc_join_password_buffer(void **state)
{
	DATA_BLOB session_key = data_blob_const("SystemLibraryDTC", 16);
	struct wkssvc_PasswordBuffer *pwd_buf = NULL;
	char *password_decoded = NULL;
	TALLOC_CTX *mem_ctx = NULL;
	WERROR werr;

	mem_ctx = talloc_new(NULL);
	assert_non_null(mem_ctx);

	werr = encode_wkssvc_join_password_buffer(mem_ctx,
						  PASSWORD,
						  &session_key,
						  &pwd_buf);
	assert_true(W_ERROR_IS_OK(werr));
	assert_non_null(pwd_buf);

	werr = decode_wkssvc_join_password_buffer(mem_ctx,
						  pwd_buf,
						  &session_key,
						  &password_decoded);
	assert_true(W_ERROR_IS_OK(werr));
	assert_non_null(password_decoded);
	assert_string_equal(password_decoded, PASSWORD);

	TALLOC_FREE(mem_ctx);
}

int main(int argc, char *argv[])
{
	int rc;
	const struct CMUnitTest tests[] = {
		cmocka_unit_test(torture_decode_rc4_passwd_buffer),
		cmocka_unit_test(torture_rc4_passwd_buffer),
		cmocka_unit_test(torture_endode_decode_rc4_passwd_buffer),
		cmocka_unit_test(torture_decode_wkssvc_join_password_buffer),
		cmocka_unit_test(torture_wkssvc_join_password_buffer),
	};

	if (argc == 2) {
		cmocka_set_test_filter(argv[1]);
	}
	cmocka_set_message_output(CM_OUTPUT_SUBUNIT);

	rc = cmocka_run_group_tests(tests, NULL, NULL);

	return rc;
}
